<?php

namespace App\Http\Controllers;

use App\Models\Venta;
use App\Models\VentaDetalle;
use App\Models\Producto;
use App\Models\Alquiler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Barryvdh\DomPDF\Facade\Pdf;

class VentaController extends Controller
{
    
    public function indexDetalle(Request $request)
    {
        $buscar = $request->input('buscar');
        $fechaInicio = $request->input('fecha_inicio');
        $fechaFin = $request->input('fecha_fin');
    
        $baseQuery = Venta::with(['detalles.producto', 'alquiler'])
        ->when($buscar, function ($query) use ($buscar) {
            $query->where('id', 'like', "%{$buscar}%")
                ->orWhereHas('alquiler', function ($q) use ($buscar) {
                    $q->where('cliente', 'like', "%{$buscar}%");
                });
        })
        
            ->when($fechaInicio, function ($query) use ($fechaInicio) {
                $query->whereDate('created_at', '>=', $fechaInicio);
            })
            ->when($fechaFin, function ($query) use ($fechaFin) {
                $query->whereDate('created_at', '<=', $fechaFin);
            });
    
        $ventas = (clone $baseQuery)->latest()->paginate(10);
        $ventasTotales = (clone $baseQuery)->sum('total');
        $ventasPagadas = (clone $baseQuery)->where('estado', 'pagado')->count();
        $ventasPendientes = (clone $baseQuery)->where('estado', 'pendiente')->count();
        $totalTransacciones = (clone $baseQuery)->count();
    
        return view('ventas.ventatotal', compact(
            'ventas',
            'ventasTotales',
            'ventasPagadas',
            'ventasPendientes',
            'totalTransacciones'
        ));
    }
    

    



    public function store(Request $request)
{
    try {
        Log::info('📦 Iniciando registro de venta...', $request->all());

        // Validar entrada sin obligar monto_pagado
        $request->validate([
            'alquiler_id' => 'required|exists:alquileres,id',
            'productos' => 'required|array',
            'productos.*.id' => 'required|integer|exists:productos,id',
            'productos.*.cantidad' => 'required|integer|min:1',
            'monto_pagado' => 'nullable|numeric|min:0',
        ]);

        $totalVenta = 0;

        foreach ($request->productos as $producto) {
            $productoModel = Producto::findOrFail($producto['id']);

            if ($productoModel->stock < $producto['cantidad']) {
                Log::warning("❌ Stock insuficiente para el producto ID: {$producto['id']}");
                return response()->json([
                    'success' => false,
                    'error' => "Stock insuficiente para el producto: {$productoModel->nombre}"
                ], 400);
            }

            $totalVenta += $producto['cantidad'] * $productoModel->precio;
        }

        // Guardar venta con monto pagado opcional
        $montoPagado = $request->monto_pagado ?? 0;
        $vuelto = $montoPagado - $totalVenta;

        $venta = Venta::create([
            'alquiler_id' => $request->alquiler_id,
            'monto_pagado' => $montoPagado,
            'vuelto' => $vuelto >= 0 ? $vuelto : 0,
            'total' => $totalVenta, // Asegúrate de tener este campo en la DB
            'estado' => $montoPagado >= $totalVenta ? 'pagado' : 'pendiente', // Este campo también debe estar
        ]);

        Log::info("✅ Venta creada: ID {$venta->id}");

        foreach ($request->productos as $producto) {
            $productoModel = Producto::findOrFail($producto['id']);
            $precioUnitario = $productoModel->precio;
            $cantidad = $producto['cantidad'];
            $subtotal = $precioUnitario * $cantidad;

            VentaDetalle::create([
                'venta_id' => $venta->id,
                'producto_id' => $productoModel->id,
                'cantidad' => $cantidad,
                'precio_unitario' => $precioUnitario,
                'subtotal' => $subtotal,
            ]);

            // Descontar stock
            $productoModel->stock -= $cantidad;
            $productoModel->save();
        }

        Log::info("🟢 Venta finalizada correctamente: Venta ID {$venta->id}");

        return response()->json(['success' => true, 'message' => 'Venta registrada sin cobro.']);

    } catch (\Exception $e) {
        Log::error('❌ Error en store(): ' . $e->getMessage(), [
            'linea' => $e->getLine(),
            'file' => $e->getFile()
        ]);

        return response()->json([
            'success' => false,
            'error' => 'Error al registrar la venta: ' . $e->getMessage()
        ], 500);
    }
}
public function cobrar(Request $request, $id)
{
    try {
        $venta = Venta::findOrFail($id);

        $request->validate([
            'monto_pagado' => 'required|numeric|min:0',
        ]);

        $total = $venta->total;

        if ($request->monto_pagado < $total) {
            return response()->json([
                'success' => false,
                'error' => 'El monto recibido no cubre el total de la venta.'
            ], 400);
        }

        $venta->monto_pagado = $request->monto_pagado;
        $venta->vuelto = $request->monto_pagado - $total;
        $venta->estado = 'pagado';
        $venta->save();

        return response()->json([
            'success' => true,
            'message' => 'Venta cobrada correctamente.'
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'error' => 'Error al cobrar la venta: ' . $e->getMessage()
        ], 500);
    }
}


    public function crear()
    {
        $alquileres = Alquiler::all();
        $productos = Producto::all();
        return view('ventas.crear', compact('alquileres', 'productos'));
    }

    public function destroy($id)
    {
        $venta = Venta::findOrFail($id);
        $venta->delete();

        return redirect()->route('ventas.index')->with('success', 'Venta eliminada correctamente.');
    }
    public function show($id)
{
    $venta = Venta::with(['detalles.producto', 'alquiler.cliente'])->findOrFail($id);
    return view('ventas.show', compact('venta'));
}
public function ticketPdf($id)
{
    $venta = Venta::with(['detalles.producto', 'alquiler'])->findOrFail($id);

    // Contenido del QR (puedes personalizarlo más si deseas)
    $qrData = "VENTA ID: {$venta->id} - TOTAL: S/" . number_format($venta->detalles->sum('subtotal'), 2);
    
    $qrUrl = 'https://api.qrserver.com/v1/create-qr-code/?size=100x100&data=' . urlencode($qrData);
    $qrImage = base64_encode(file_get_contents($qrUrl));
    $qrSrc = 'data:image/png;base64,' . $qrImage;

    $pdf = PDF::loadView('ventas.ticket-pdf', compact('venta', 'qrSrc'));
    $pdf->setPaper([0, 0, 226.77, 700], 'portrait'); // Ticket tamaño térmico

    return $pdf->stream("ticket_venta_{$venta->id}.pdf");
}


}