@extends('adminlte::page')

@section('title', 'Punto de Venta')

@section('css')
<link href="https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/css/select2.min.css" rel="stylesheet" />
<link href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css" rel="stylesheet" />
<link href="https://cdnjs.cloudflare.com/ajax/libs/sweetalert2/11.4.8/sweetalert2.min.css" rel="stylesheet" />
<style>
    :root {
        --primary-color: #4361ee;
        --secondary-color: #3bc14a;
        --accent-color: #f72585;
        --danger-color: #e63946;
        --light-bg: #f8f9fa;
        --dark-bg: #212529;
        --card-shadow: 0 8px 16px rgba(0, 0, 0, 0.12);
        --transition: all 0.3s cubic-bezier(0.25, 0.8, 0.25, 1);
        --border-radius: 12px;
        --text-primary: #333;
        --text-secondary: #6c757d;
    }
    
    body {
        background-color: #f0f2f5;
        font-family: 'Poppins', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }
    
    .pos-container {
        padding: 20px;
    }
    
    .card-main {
        border: none;
        border-radius: var(--border-radius);
        box-shadow: var(--card-shadow);
        transition: var(--transition);
        overflow: hidden;
        background-color: white;
        margin-bottom: 24px;
    }
    
    .card-main:hover {
        transform: translateY(-5px);
        box-shadow: 0 12px 24px rgba(0, 0, 0, 0.15);
    }
    
    .card-header-custom {
        background: linear-gradient(120deg, #ffffff, #f9fafb);
        border-bottom: 1px solid rgba(0, 0, 0, 0.05);
        padding: 18px 22px;
        display: flex;
        align-items: center;
    }
    
    .title-section {
        color: var(--primary-color);
        font-weight: 700;
        margin-bottom: 0;
        font-size: 1.35rem;
        letter-spacing: -0.5px;
    }
    
    .title-section i {
        background: linear-gradient(45deg, var(--primary-color), #4895ef);
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        margin-right: 10px;
    }
    
    /* Productos */
    .products-container {
        max-height: 72vh;
        overflow-y: auto;
        padding-right: 10px;
        padding-left: 5px;
    }

    .products-container::-webkit-scrollbar {
        width: 6px;
    }

    .products-container::-webkit-scrollbar-track {
        background: #f1f1f1;
        border-radius: 10px;
    }

    .products-container::-webkit-scrollbar-thumb {
        background: #c1c1c1;
        border-radius: 10px;
    }

    .products-container::-webkit-scrollbar-thumb:hover {
        background: #a8a8a8;
    }

    .product-card {
        background: white;
        border-radius: 16px;
        overflow: hidden;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
        transition: var(--transition);
        height: 100%;
        border: 1px solid rgba(0, 0, 0, 0.04);
    }
    
    .product-card:hover {
        transform: translateY(-5px) scale(1.02);
        box-shadow: 0 10px 20px rgba(0, 0, 0, 0.12);
    }
    
    .product-img-container {
    position: relative;
    height: 180px; /* antes 160px */
    overflow: hidden;
}
    .product-img {
        height: 100%;
        width: 100%;
        object-fit: cover;
        transition: transform 0.5s ease;
    }
    
    .product-card:hover .product-img {
        transform: scale(1.08);
    }
    
    .product-stock-badge {
        position: absolute;
        top: 10px;
        right: 10px;
        padding: 5px 10px;
        border-radius: 20px;
        font-size: 0.75rem;
        font-weight: 600;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        z-index: 5;
    }
    
    .stock-available {
        background-color: rgba(46, 213, 115, 0.85);
        color: white;
    }
    
    .stock-out {
        background-color: rgba(235, 59, 90, 0.85);
        color: white;
    }
    
    .product-info {
        padding: 15px;
    }
    
    .product-title {
        font-size: 1.1rem;
        font-weight: 600;
        color: var(--text-primary);
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
        margin-bottom: 8px;
    }
    
    .product-price {
        color: var(--primary-color);
        font-weight: 700;
        font-size: 1.25rem;
        display: block;
        margin-bottom: 12px;
    }
    
    .btn-add-cart {
        background: linear-gradient(45deg, var(--secondary-color), #4cc94e);
        border: none;
        border-radius: 30px;
        padding: 10px 20px;
        font-weight: 600;
        color: white;
        transition: all 0.3s;
        width: 100%;
        box-shadow: 0 4px 8px rgba(46, 213, 115, 0.2);
    }
    
    .btn-add-cart:hover {
        background: linear-gradient(45deg, #4cc94e, #5dda60);
        transform: translateY(-2px);
        box-shadow: 0 6px 12px rgba(46, 213, 115, 0.3);
    }
    
    .btn-add-cart:disabled {
        background: #e2e2e2;
        color: #999;
        box-shadow: none;
        cursor: not-allowed;
    }
    
    /* Carrito */
    .cart-section {
        background: white;
        border-radius: var(--border-radius);
        box-shadow: var(--card-shadow);
        transition: var(--transition);
        height: 100%;
    }
    
    .cart-container {
        max-height: 40vh;
        overflow-y: auto;
        padding: 10px;
    }
    
    .cart-container::-webkit-scrollbar {
        width: 6px;
    }
    
    .cart-container::-webkit-scrollbar-track {
        background: #f1f1f1;
        border-radius: 10px;
    }
    
    .cart-container::-webkit-scrollbar-thumb {
        background: #c1c1c1;
        border-radius: 10px;
    }
    
    .cart-item {
        background: #f8fafd;
        border-left: 4px solid var(--secondary-color);
        margin-bottom: 12px;
        border-radius: 10px;
        padding: 14px;
        transition: var(--transition);
        animation: fadeIn 0.3s ease-in-out;
    }
    
    .cart-item:hover {
        background-color: #f0f4f9;
        transform: translateX(4px);
    }
    
    .cart-item-title {
        font-weight: 600;
        color: var(--text-primary);
        margin-bottom: 5px;
    }
    
    .cart-item-price {
        font-size: 0.9rem;
        color: var(--text-secondary);
    }
    
    .cart-item-subtotal {
        font-weight: 700;
        color: var(--primary-color);
    }
    
    .btn-remove-item {
        background-color: rgba(230, 57, 70, 0.1);
        border: none;
        color: var(--danger-color);
        border-radius: 6px;
        width: 32px;
        height: 32px;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.2s;
    }
    
    .btn-remove-item:hover {
        background-color: var(--danger-color);
        color: white;
    }
    
    .empty-cart {
        text-align: center;
        padding: 30px 0;
        color: var(--text-secondary);
    }
    
    .empty-cart i {
        font-size: 3rem;
        margin-bottom: 15px;
        color: #d1d1d1;
    }
    
    .total-display {
        background: linear-gradient(to right, #f8f9fa, #f0f4f9);
        padding: 18px;
        border-radius: 10px;
        font-size: 1.3rem;
        font-weight: 600;
        margin-top: 15px;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
    }
    
    .total-label {
        color: var(--text-primary);
    }
    
    .total-amount {
        color: var(--primary-color);
        font-weight: 700;
    }
    
    /* Payment section */
    .payment-section {
        background-color: white;
        border-radius: var(--border-radius);
        padding: 20px;
    }
    
    .form-label {
        font-weight: 600;
        color: var(--text-primary);
        margin-bottom: 8px;
    }
    
    .input-payment {
        border-radius: 8px;
        padding: 12px 15px;
        border: 1px solid #ddd;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.03);
        font-size: 1.1rem;
        transition: all 0.2s;
    }
    
    .input-payment:focus {
        border-color: var(--primary-color);
        box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.15);
    }
    
    .input-group-text {
        background-color: #f8f9fa;
        border-color: #ddd;
        color: var(--text-secondary);
        font-weight: 600;
    }
    
    .change-display {
        background-color: #f0f9f0;
        padding: 14px;
        border-radius: 8px;
        margin-top: 15px;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    
    .change-label {
        font-weight: 600;
        color: var(--text-primary);
    }
    
    .change-amount {
        font-weight: 700;
        color: var(--secondary-color);
    }
    
    .change-amount.negative {
        color: var(--danger-color);
    }
    
    /* Customer section */
    .select2-container--default .select2-selection--single {
        height: 50px;
        border-radius: 8px;
        border: 1px solid #ddd;
        padding: 10px 15px;
        font-size: 1.1rem;
    }
    
    .select2-container--default .select2-selection--single .select2-selection__arrow {
        height: 48px;
        right: 10px;
    }
    
    .select2-dropdown {
        border-radius: 8px;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
        border: 1px solid #eaeaea;
    }
    
    .select2-container--default .select2-results__option--highlighted[aria-selected] {
        background-color: var(--primary-color);
    }
    
    /* Search bar */
    .search-box {
        position: relative;
        margin-bottom: 25px;
    }
    
    .search-input {
        padding: 14px 20px 14px 50px;
        border-radius: 30px;
        border: 1px solid #eaeaea;
        font-size: 1.1rem;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
        transition: all 0.3s;
    }
    
    .search-input:focus {
        box-shadow: 0 6px 16px rgba(0, 0, 0, 0.1);
        border-color: var(--primary-color);
    }
    
    .search-icon {
        position: absolute;
        top: 15px;
        left: 20px;
        color: var(--text-secondary);
        font-size: 1.2rem;
    }
    
    /* Bottom buttons */
    .btn-checkout {
        background: linear-gradient(45deg, var(--primary-color), #4895ef);
        border: none;
        padding: 16px;
        font-size: 1.2rem;
        font-weight: 700;
        border-radius: 12px;
        color: white;
        transition: all 0.3s;
        box-shadow: 0 6px 16px rgba(67, 97, 238, 0.3);
        letter-spacing: 0.5px;
    }
    
    .btn-checkout:hover:not(:disabled) {
        transform: translateY(-3px);
        box-shadow: 0 8px 24px rgba(67, 97, 238, 0.4);
    }
    
    .btn-checkout:active:not(:disabled) {
        transform: translateY(1px);
    }
    
    .btn-checkout:disabled {
        background: linear-gradient(45deg, #a8b1e6, #b5c5f1);
        cursor: not-allowed;
        box-shadow: none;
    }
    
    .btn-checkout i {
        margin-right: 10px;
    }
    
    /* Product categories - New feature */
    .category-pills {
        display: flex;
        overflow-x: auto;
        padding: 10px 0;
        margin-bottom: 20px;
        -ms-overflow-style: none;
        scrollbar-width: none;
    }
    
    .category-pills::-webkit-scrollbar {
        display: none;
    }
    
    .category-pill {
        padding: 8px 18px;
        background-color: #f0f2f5;
        border-radius: 30px;
        margin-right: 10px;
        font-weight: 500;
        font-size: 0.9rem;
        color: var(--text-secondary);
        white-space: nowrap;
        cursor: pointer;
        transition: all 0.2s;
        border: 1px solid transparent;
    }
    
    .category-pill:hover, .category-pill.active {
        background-color: white;
        color: var(--primary-color);
        border-color: var(--primary-color);
        box-shadow: 0 2px 8px rgba(67, 97, 238, 0.15);
    }
    
    /* Quantity controls - New feature */
    .quantity-control {
        display: flex;
        align-items: center;
        justify-content: center;
        margin-top: 8px;
    }
    
    .btn-quantity {
        width: 28px;
        height: 28px;
        border-radius: 50%;
        border: none;
        background-color: #f0f4f9;
        color: var(--text-primary);
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: bold;
        transition: all 0.2s;
    }
    
    .btn-quantity:hover {
        background-color: var(--primary-color);
        color: white;
    }
    
    .quantity-value {
        width: 40px;
        text-align: center;
        font-weight: 600;
        font-size: 1rem;
    }
    
    /* Animations */
    @keyframes fadeIn {
        from { opacity: 0; transform: translateY(10px); }
        to { opacity: 1; transform: translateY(0); }
    }
    
    @keyframes pulse {
        0% { transform: scale(1); }
        50% { transform: scale(1.05); }
        100% { transform: scale(1); }
    }
    
    .pulse-animation {
        animation: pulse 1s infinite;
    }
    
    /* Responsive */
    @media (max-width: 992px) {
        .pos-container {
            padding: 15px;
        }
        
        .product-img-container {
            height: 140px;
        }
        
        .products-container {
            max-height: 60vh;
        }
        
        .cart-container {
            max-height: 30vh;
        }
        
        .btn-checkout {
            padding: 14px;
            font-size: 1.1rem;
        }
    }
    
    @media (max-width: 768px) {
        .pos-container {
            padding: 10px;
        }
        
        .product-img-container {
            height: 120px;
        }
        
        .products-container {
            max-height: 50vh;
        }
        
        .title-section {
            font-size: 1.2rem;
        }
        
        .product-title {
            font-size: 1rem;
        }
        
        .product-price {
            font-size: 1.1rem;
        }
    }
</style>
@endsection

@section('content')
<div class="pos-container">
    <div class="row">
        <!-- Columna izquierda: Productos -->
        <div class="col-lg-8 mb-4">
            <div class="card-main">
                <div class="card-header-custom">
                    <h2 class="title-section"><i class="fas fa-cash-register"></i>Punto de Venta</h2>
                </div>
                <div class="card-body">
                    <!-- Barra de búsqueda -->
                    <div class="search-box">
                        <i class="fas fa-search search-icon"></i>
                        <input type="text" id="buscarProducto" class="form-control search-input" placeholder="¿Qué producto estás buscando?">
                    </div>
                    
                    <!-- Nueva sección: Categorías de productos -->
                    <div class="category-pills">
                        <div class="category-pill active" data-category="all">Todos</div>
                        <!-- Aquí pueden ir las categorías dinámicas, por ejemplo: -->
                        <div class="category-pill" data-category="bebidas">Bebidas</div>
                        <div class="category-pill" data-category="comida">Comida</div>
                        <div class="category-pill" data-category="snacks">Snacks</div>
                        <div class="category-pill" data-category="postres">Postres</div>
                        <div class="category-pill" data-category="otros">Otros</div>
                    </div>
                    
                    <!-- Lista de productos con scroll -->
                    <div class="products-container">
                        <div class="row">
                            @foreach ($productos->sortByDesc('stock') as $producto)
                            <div class="col-lg-4 col-md-6 mb-4 product-item animate__animated animate__fadeIn" 
                                 data-category="{{ $producto->categoria ?? 'otros' }}">
                                <div class="product-card">
                                    <div class="product-img-container">
                                        <img src="{{ asset('storage/' . $producto->imagen) }}" class="product-img" alt="{{ $producto->nombre }}">
                                        <span class="product-stock-badge {{ $producto->stock > 0 ? 'stock-available' : 'stock-out' }}">
                                            {{ $producto->stock > 0 ? 'Stock: ' . $producto->stock : 'Agotado' }}
                                        </span>
                                    </div>
                                    <div class="product-info">
                                        <h5 class="product-title">{{ $producto->nombre }}</h5>
                                        <span class="product-price">S/ {{ number_format($producto->precio, 2) }}</span>
                                        <button type="button" class="btn btn-add-cart" 
                                            data-id="{{ $producto->id }}" 
                                            data-nombre="{{ $producto->nombre }}" 
                                            data-precio="{{ $producto->precio }}" 
                                            data-stock="{{ $producto->stock }}"
                                            {{ $producto->stock <= 0 ? 'disabled' : '' }}
                                            onclick="agregarAlCarrito(this)">
                                            <i class="fas fa-cart-plus mr-2"></i>Agregar
                                        </button>
                                    </div>
                                </div>
                            </div>
                            @endforeach
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Columna derecha: Carrito y Pago -->
        <div class="col-lg-4">
            <!-- Carrito de compras -->
            <div class="card-main">
                <div class="card-header-custom">
                    <h3 class="title-section"><i class="fas fa-shopping-cart"></i>Carrito de Compra</h3>
                </div>
                <div class="card-body">
                    <div id="carrito" class="cart-container">
                        <!-- Los productos agregados aparecerán aquí -->
                        <div class="empty-cart" id="empty-cart-message">
                            <i class="fas fa-shopping-cart"></i>
                            <p>Tu carrito está vacío</p>
                            <small>Los productos que agregues aparecerán aquí</small>
                        </div>
                    </div>
                    
                    <div class="total-display d-flex justify-content-between align-items-center">
                        <span class="total-label">Total:</span>
                        <span class="total-amount">S/ <span id="total_venta">0.00</span></span>
                    </div>
                </div>
            </div>
            
           
            
            <!-- Selección de cliente -->
            <div class="card-main">
                <div class="card-header-custom">
                    <h3 class="title-section"><i class="fas fa-user"></i>Datos del Cliente</h3>
                </div>
                <div class="card-body">
                    <div class="form-group">
                        <label for="alquiler_id" class="form-label">Seleccionar Cliente</label>
                        <select name="alquiler_id" id="alquiler_id" class="form-control select2" required>
                            <option value="">-- Selecciona un cliente --</option>
                            @foreach ($alquileres as $alquiler)
                                <option value="{{ $alquiler->id }}">{{ $alquiler->cliente }}</option>
                            @endforeach
                        </select>
                    </div>
                </div>
            </div>
            
            <!-- Botón de confirmación -->
            <button id="realizar_venta" class="btn btn-checkout w-100" disabled>
                <i class="fas fa-check-circle"></i>Confirmar Venta
            </button>
        </div>
    </div>
</div>
@endsection

@section('js')
<!-- jQuery y Select2 -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/js/select2.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<script>
// Inicializar Select2
$(document).ready(function() {
    $('.select2').select2({
        placeholder: "Selecciona un cliente",
        allowClear: true
    });

    // Activar el botón si cambia el cliente
    $('#alquiler_id').on('change', function () {
        actualizarTotal();
    });
});

// Variables para gestionar el carrito
let productosVenta = [];
let totalVenta = 0;

// Función de búsqueda
$('#buscarProducto').on('input', function() {
    const searchTerm = $(this).val().toLowerCase();
    
    $('.product-item').each(function() {
        const productName = $(this).find('.product-title').text().toLowerCase();
        const shouldShow = productName.includes(searchTerm);
        
        $(this).toggle(shouldShow);
    });
});

// Función para actualizar el total
function actualizarTotal() {
    totalVenta = productosVenta.reduce((sum, p) => sum + p.subtotal, 0);
    document.getElementById('total_venta').textContent = totalVenta.toFixed(2);

    // Activar botón solo si hay productos Y cliente seleccionado
    const botonVenta = document.getElementById('realizar_venta');
    const clienteSeleccionado = document.getElementById('alquiler_id').value;

    botonVenta.disabled = !(productosVenta.length > 0 && clienteSeleccionado);
}

// Función para agregar productos al carrito
function agregarAlCarrito(button) {
    let productoId = button.getAttribute('data-id');
    let nombre = button.getAttribute('data-nombre');
    let precio = parseFloat(button.getAttribute('data-precio'));
    let stock = parseInt(button.getAttribute('data-stock'));
    let cantidad = 1;

    if (cantidad > stock) {
        Swal.fire({
            icon: 'warning',
            title: 'Stock insuficiente',
            text: 'No hay suficiente stock disponible para este producto',
            confirmButtonColor: '#4361ee'
        });
        return;
    }

    let productoExistente = productosVenta.find(p => p.id == productoId);
    if (productoExistente) {
        productoExistente.cantidad += cantidad;
        productoExistente.subtotal = productoExistente.cantidad * productoExistente.precio;
    } else {
        productosVenta.push({ 
            id: productoId, 
            nombre, 
            cantidad, 
            precio, 
            subtotal: cantidad * precio,
            stock: stock
        });
    }

    renderCarrito();
    actualizarTotal();
}

// Función para renderizar el carrito
function renderCarrito() {
    let carrito = document.getElementById('carrito');
    
    if (productosVenta.length === 0) {
        carrito.innerHTML = `
            <div class="empty-cart" id="empty-cart-message">
                <i class="fas fa-shopping-cart"></i>
                <p>Tu carrito está vacío</p>
                <small>Los productos que agregues aparecerán aquí</small>
            </div>`;
        return;
    }

    let html = '';
    productosVenta.forEach((p, index) => {
        html += `
        <div class="cart-item animate__animated animate__fadeIn">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h6 class="cart-item-title">${p.nombre}</h6>
                    <div class="d-flex align-items-center">
                        <span class="cart-item-price">S/ ${p.precio.toFixed(2)} × </span>
                        <div class="quantity-control ml-2">
                            <button type="button" class="btn-quantity" onclick="decrementarCantidad(${index})" ${p.cantidad <= 1 ? 'disabled' : ''}>-</button>
                            <span class="quantity-value mx-2">${p.cantidad}</span>
                            <button type="button" class="btn-quantity" onclick="incrementarCantidad(${index})" ${p.cantidad >= p.stock ? 'disabled' : ''}>+</button>
                        </div>
                    </div>
                </div>
                <div class="d-flex align-items-center">
                    <span class="cart-item-subtotal mr-3">S/ ${p.subtotal.toFixed(2)}</span>
                    <button class="btn-remove-item" onclick="eliminarProducto(${index})">
                        <i class="fas fa-trash-alt"></i>
                    </button>
                </div>
            </div>
        </div>`;
    });
    
    carrito.innerHTML = html;
}

// Función para incrementar cantidad
function incrementarCantidad(index) {
    const producto = productosVenta[index];
    const stockBtn = document.querySelector(`button[data-id="${producto.id}"]`);
    const stock = stockBtn ? parseInt(stockBtn.getAttribute('data-stock')) : producto.stock;
    
    if (producto.cantidad < stock) {
        producto.cantidad++;
        producto.subtotal = producto.cantidad * producto.precio;
        renderCarrito();
        actualizarTotal();
    } else {
        Swal.fire({
            icon: 'warning',
            title: 'Stock máximo alcanzado',
            text: 'No hay suficiente stock disponible para agregar más unidades',
            confirmButtonColor: '#4361ee'
        });
    }
}

// Función para decrementar cantidad
function decrementarCantidad(index) {
    const producto = productosVenta[index];
    
    if (producto.cantidad > 1) {
        producto.cantidad--;
        producto.subtotal = producto.cantidad * producto.precio;
        renderCarrito();
        actualizarTotal();
    }
}

// Función para eliminar producto del carrito
function eliminarProducto(index) {
    productosVenta.splice(index, 1);
    renderCarrito();
    actualizarTotal();
}

// Confirmar la venta
document.getElementById('realizar_venta').addEventListener('click', async function() {
    if (productosVenta.length === 0) {
        Swal.fire({
            icon: 'warning',
            title: 'Carrito vacío',
            text: 'Agrega productos al carrito antes de realizar la venta',
            confirmButtonColor: '#4361ee'
        });
        return;
    }

    let alquilerId = document.getElementById('alquiler_id').value;
    if (!alquilerId) {
        Swal.fire({
            icon: 'error',
            title: 'Cliente requerido',
            text: 'Selecciona un cliente antes de realizar la venta',
            confirmButtonColor: '#4361ee'
        });
        return;
    }

    const { value: confirmacion } = await Swal.fire({
        title: '¿Confirmar venta?',
        text: `Total: S/ ${totalVenta.toFixed(2)}`,
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#4361ee',
        cancelButtonColor: '#6c757d',
        confirmButtonText: 'Sí, confirmar',
        cancelButtonText: 'Cancelar'
    });

    if (!confirmacion) return;

    try {
        const response = await fetch('/ventas/store', {
            method: 'POST',
            headers: { 
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify({
                productos: productosVenta,
                alquiler_id: alquilerId
            })
        });

        const data = await response.json();

        if (data.success) {
            await Swal.fire({
                title: '¡Venta registrada!',
                text: 'Puedes realizar el cobro desde el historial de ventas.',
                icon: 'success',
                confirmButtonColor: '#4361ee'
            });

            productosVenta = [];
            document.getElementById('alquiler_id').value = '';
            $('.select2').val(null).trigger('change');
            renderCarrito();
            actualizarTotal();
        } else {
            throw new Error(data.error || 'Error desconocido');
        }
    } catch (error) {
        console.error('Error:', error);
        Swal.fire({
            title: 'Error',
            text: error.message || 'Ocurrió un error al procesar la venta',
            icon: 'error',
            confirmButtonColor: '#4361ee'
        });
    }
});

// Manejo de categorías
$('.category-pill').click(function() {
    const category = $(this).data('category');
    
    $('.category-pill').removeClass('active');
    $(this).addClass('active');
    
    if (category === 'all') {
        $('.product-item').show();
    } else {
        $('.product-item').each(function() {
            const productCategory = $(this).data('category');
            $(this).toggle(productCategory === category);
        });
    }
});
</script>

@endsection