@extends('adminlte::page')

@section('title', 'Detalle de Ventas')

@section('content')
<div class="container-fluid mt-3">
    <div class="card shadow">
        <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
            <h3 class="card-title mb-0">
                <i class="fas fa-cash-register mr-2"></i> Listado de Ventas
            </h3>
            <a href="{{ route('ventas.index') }}" class="btn btn-light btn-sm">
                <i class="fas fa-arrow-left mr-1"></i> Volver
            </a>
        </div>
        
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-bordered table-hover table-striped mb-0">
                    <thead class="thead-dark">
                        <tr>
                            <th width="5%">ID</th>
                            <th width="10%">Cliente</th>
                            <th width="20%">Productos</th>
                            <th width="12%">Fecha y Hora</th>
                            <th width="10%" class="text-right">Monto Pagado</th>
                            <th width="10%" class="text-right">Vuelto</th>
                            <th width="10%" class="text-right">Subtotal</th>
                            <th width="15%" class="text-center">Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($ventas as $venta)
                        <tr>
                            <td class="align-middle text-center">{{ $venta->id }}</td>
                            <td class="align-middle">{{ $venta->alquiler->cliente ?? 'Venta directa' }}</td>
                            <td>
                                <!-- Listado horizontal de productos mejorado -->
                                @if($venta->detalles->count())
                                    <div class="product-list">
                                        @foreach($venta->detalles as $detalle)
                                            <span class="product-badge">
                                                {{ $detalle->producto->nombre ?? $detalle->nombre_producto ?? 'Producto eliminado' }}
                                                <small class="product-quantity">×{{ $detalle->cantidad }}</small>
                                            </span>
                                        @endforeach
                                    </div>
                                @else
                                    <span class="text-muted font-italic">Sin productos</span>
                                @endif
                            </td>
                            <td class="align-middle">{{ $venta->created_at->format('d/m/Y H:i') }}</td>
                            <td class="align-middle text-right">S/ {{ number_format($venta->monto_pagado, 2) }}</td>
                            <td class="align-middle text-right">S/ {{ number_format($venta->vuelto, 2) }}</td>
                            <td class="align-middle text-right font-weight-bold">
                                S/ {{ number_format($venta->detalles->sum('subtotal'), 2) }}
                            </td>
                            <td class="align-middle text-center">
                                <div class="btn-group">
                                    <button class="btn btn-sm btn-info" data-toggle="modal" data-target="#modalVenta{{ $venta->id }}" title="Ver detalle">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                    <form action="{{ route('ventas.destroy', $venta->id) }}" method="POST" style="display:inline;">
                                        @csrf
                                        @method('DELETE')
                                        <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('¿Está seguro de eliminar esta venta?')" title="Eliminar">
                                            <i class="fas fa-trash-alt"></i>
                                        </button>
                                    </form>
                                    <button class="btn btn-sm btn-success btn-print" data-venta-id="{{ $venta->id }}" title="Imprimir">
                                        <i class="fas fa-print"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                        @empty
                        <tr>
                            <td colspan="8" class="text-center py-5">
                                <i class="fas fa-info-circle fa-3x text-muted mb-3"></i>
                                <p class="mb-0 text-muted">No se encontraron ventas registradas</p>
                            </td>
                        </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>

        @if($ventas->hasPages())
        <div class="card-footer">
            {{ $ventas->links() }}
        </div>
        @endif
    </div>
</div>

<!-- Modales fuera del bucle para mejor rendimiento -->
@foreach($ventas as $venta)
<div class="modal fade" id="modalVenta{{ $venta->id }}" tabindex="-1" role="dialog" aria-labelledby="modalVentaLabel{{ $venta->id }}" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header bg-info text-white">
                <h5 class="modal-title">
                    <i class="fas fa-file-invoice mr-2"></i> Detalle de Venta #{{ $venta->id }} - {{ $venta->created_at->format('d/m/Y H:i') }}
                </h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <div class="row mb-3 border-bottom pb-3">
                    <div class="col-md-6">
                        <h6 class="text-muted mb-1">Cliente</h6>
                        <p class="font-weight-bold mb-0">{{ optional($venta->alquiler)->cliente->nombre ?? 'Venta directa' }}</p>
                    </div>
                    <div class="col-md-6 text-right">
                        <h6 class="text-muted mb-1">Total</h6>
                        <p class="font-weight-bold mb-0 text-primary">S/ {{ number_format($venta->detalles->sum('subtotal'), 2) }}</p>
                    </div>
                </div>
                
                @if($venta->detalles->count())
                <div class="table-responsive">
                    <table class="table table-sm table-bordered table-hover">
                        <thead class="bg-light">
                            <tr>
                                <th width="50%">Producto</th>
                                <th width="15%" class="text-center">Cantidad</th>
                                <th width="15%" class="text-right">P. Unitario</th>
                                <th width="20%" class="text-right">Subtotal</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($venta->detalles as $detalle)
                            <tr>
                                <td>{{ $detalle->producto->nombre ?? $detalle->nombre_producto ?? 'Producto eliminado' }}</td>
                                <td class="text-center">{{ $detalle->cantidad }}</td>
                                <td class="text-right">S/ {{ number_format($detalle->precio_unitario, 2) }}</td>
                                <td class="text-right">S/ {{ number_format($detalle->subtotal, 2) }}</td>
                            </tr>
                            @endforeach
                            <tr class="bg-light">
                                <td colspan="3" class="text-right font-weight-bold">Total:</td>
                                <td class="text-right font-weight-bold">S/ {{ number_format($venta->detalles->sum('subtotal'), 2) }}</td>
                            </tr>
                        </tbody>
                    </table>
                </div>
                @else
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle mr-2"></i> No hay productos registrados en esta venta.
                </div>
                @endif
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">
                    <i class="fas fa-times mr-1"></i> Cerrar
                </button>
                <button type="button" class="btn btn-primary btn-print" data-venta-id="{{ $venta->id }}">
                    <i class="fas fa-print mr-1"></i> Imprimir
                </button>
            </div>
        </div>
    </div>
</div>
@endforeach
@endsection

@push('css')
<style>
    /* Estilos generales */
    .table thead th {
        vertical-align: middle;
        font-weight: 600;
    }
    .btn-sm {
        padding: 0.25rem 0.5rem;
        font-size: 0.85rem;
        line-height: 1.5;
        border-radius: 3px;
    }
    
    /* Estilos para la lista de productos */
    .product-list {
        display: flex;
        flex-wrap: wrap;
        gap: 4px;
        max-width: 100%;
    }
    
    .product-badge {
        display: inline-block;
        padding: 3px 8px;
        margin-bottom: 3px;
        background-color: #f8f9fa;
        border: 1px solid #e9ecef;
        border-radius: 4px;
        font-size: 0.8rem;
        color: #495057;
        box-shadow: 0 1px 2px rgba(0,0,0,0.05);
    }
    
    .product-badge:hover {
        background-color: #e9ecef;
    }
    
    .product-quantity {
        display: inline-block;
        color: #6c757d;
        margin-left: 3px;
        font-weight: 500;
    }
    
    /* Mejoras para el modal */
    .modal-header .modal-title {
        font-size: 1.1rem;
    }
    
    /* Estilos para botones de acción */
    .btn-group .btn:not(:last-child) {
        margin-right: 2px;
    }
    
    /* Mejora para la tabla principal */
    .table-striped tbody tr:nth-of-type(odd) {
        background-color: rgba(0,0,0,.02);
    }
    
    .card.shadow {
        box-shadow: 0 .15rem 1.75rem 0 rgba(58,59,69,.15)!important;
    }
</style>
@endpush

@push('js')
<script>
    $(document).ready(function () {
        // Configuración para imprimir
        $('.btn-print').on('click', function () {
            const ventaId = $(this).data('venta-id');
            const modalContent = $('#modalVenta'+ventaId+' .modal-body').html();
            const ventaTitle = $('#modalVenta'+ventaId+' .modal-title').text();
            
            const printWindow = window.open('', '_blank');
            printWindow.document.write(`
                <html>
                    <head>
                        <title>${ventaTitle}</title>
                        <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
                        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.1/css/all.min.css">
                        <style>
                            body { 
                                padding: 20px; 
                                font-family: Arial, sans-serif;
                            }
                            .table { 
                                width: 100%; 
                                margin-bottom: 1rem; 
                                border-collapse: collapse;
                            }
                            .table th, .table td { 
                                padding: 0.75rem; 
                                border: 1px solid #dee2e6;
                            }
                            .table thead th {
                                background-color: #f8f9fa;
                                color: #495057;
                                border-bottom: 2px solid #dee2e6;
                            }
                            .text-right { text-align: right; }
                            .text-center { text-align: center; }
                            .no-print { display: none; }
                            .header-print {
                                text-align: center;
                                margin-bottom: 20px;
                                padding-bottom: 15px;
                                border-bottom: 1px solid #dee2e6;
                            }
                            @media print {
                                .no-print { display: none; }
                                body { padding: 0; }
                            }
                        </style>
                    </head>
                    <body>
                        <div class="header-print">
                            <h4>${ventaTitle}</h4>
                            <p class="text-muted">Documento generado el ${new Date().toLocaleString()}</p>
                        </div>
                        <div class="container mt-4">
                            ${modalContent}
                        </div>
                        <script>
                            window.onload = function() {
                                window.print();
                                setTimeout(function() {
                                    window.close();
                                }, 1000);
                            }
                        <\/script>
                    </body>
                </html>
            `);
            printWindow.document.close();
        });
    });
</script>
@endpush